;;; ini-mode.el --- major mode for editing ini files

;; Copyright (C) 2000 Jan Borsodi

;; Author: Jan Borsodi <jb@ez.no>
;; Maintainer: jb@ez.no
;; Keywords: INI
;; Created: 2000-07-01
;; Modified: 2000-07-18

(defconst ini-version "0.2"
  "INI Mode version number.")

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

;;; Usage

;; Place this file in your Emacs lisp path (eg. site-lisp)
;; and add to your .emacs file:
;; (require 'ini-mode)

;;; Commentary

;; INI mode is a major mode for editing INI files used in the Windows and Linux world.

;; This mode officially supports only GNU Emacs. If you have have any
;; problems with it in XEmacs and/or has a tips on how to make it
;; work please report to me.

;;; Changelog

;; 0.1
;; 
;; Initial version, very basic

;; 0.2
;; 
;; Supports arrays []
;; Activates for files ending with .ini.php and .ini.append.php

;; Requirements

(require 'font-lock)
(require 'custom)

;; The code

(defgroup ini nil
  "Highlight or hide text according to ini conditionals."
  :group 'text
  :prefix "ini-")

(defcustom ini-mode-hook nil
  "*Hook called by `ini-mode'."
  :type 'hook
  :group 'ini)

(defvar ini-mode-map ()
  "Keymap used in ini mode.")
(when (not ini-mode-map)
  (setq ini-mode-map (make-sparse-keymap))
;;  (define-key ini-mode-map (read-kbd-macro "M-TAB") 'ini-complete-property)
)

(defvar ini-font-lock-keywords
  (list
   ;; Groups
   '("^\\(\\[\\)\\([^]]+\\)\\(\\]\\)"
     (0 font-lock-keyword-face t t)
     (2 font-lock-variable-name-face t t))
   ;; Variable assignment
   '("^[ \t]*\\([A-Za-z0-9_-]+\\)\\(\\[\\([A-Za-z0-9_-]*\\)\\]\\)?\\(\\(=\\)\\(.*\\)\\)?$"
     (1 font-lock-function-name-face t t)
     (2 font-lock-keyword-face t t)
     (3 font-lock-function-name-face t t)
     (5 font-lock-keys-face t t)
     (6 font-lock-string-face t t))
   '("\\<[0-9]+\\>" (0 font-lock-number-face t t)) ;; Numbers
   '("[0-9]+:[0-9]+" (0 font-lock-floatnumber-face t t)) ;; Time
   '("\\<[a-zA-Z0-9]+@[a-zA-Z0-9]+\\(\\.[a-zA-Z0-9]+\\)*\\>" (0 font-lock-constant-face t t)) ;; email
   '("\\([a-zA-Z]:[/\\]\\|[/\\]\\)[a-zA-Z0-9.]+\\([/\\][a-zA-Z0-9.]+\\)*" (0 font-lock-type-face t t)) ;; Directory
   '("\\([a-zA-Z0-9.]+[/\\]\\)+[a-zA-Z0-9.]*" (0 font-lock-type-face t t)) ;; Directory
   '("[a-zA-Z0-9._<>-]*[;,][a-zA-Z0-9._<>-]*\\([;,][a-zA-Z0-9._<>-]*\\)*" ;; Lists
     (0 font-lock-hexnumber-face t t))
   '("[a-z]{2}_[A-Z]{2}" (0 font-lock-reference-face t t)) ;; Language
   '("\\([a-z]+\\)://\\([a-zA-Z0-9_]+\\(\\.[a-zA-Z0-9_]+\\)*\\)[^\n]+" ;; URL
     (0 font-lock-keyword-face t t)
     (1 font-lock-type-face t t)
     (2 font-lock-variable-name-face t t)
     (3 font-lock-string-face t t))
   (list (concat "\\<" (regexp-opt '("enable" "enabled" "disable" "disabled" "true" "false" "none" "on" "off") t) "\\>") '(0 font-lock-builtin-face t t)) ;; Builtins
   '("\\(^#\\|##\\)\\([?][^?]+[?]\\)?.*$" 
     (0 font-lock-comment-face t t) ;; Comments
     (2 font-lock-function-name-face t t)) ;; Comments
   )
  "Rules for highlighting ini files.")

;;;###autoload
(defun ini-mode()
  "Major mode for editing ini files.\n\n\\{ini-mode-map}"
  (interactive)

  (kill-all-local-variables)

  ;; Indentation
;   (make-local-variable 'indent-line-function)
;   (setq indent-line-function 'ini-indent-line)

  ; font-locking
  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults '(ini-font-lock-keywords nil t nil nil))

  ; Setting up typing shortcuts
  (make-local-variable 'skeleton-end-hook)
  (setq skeleton-end-hook nil)

  (setq case-fold-search t)

  (turn-on-font-lock)

  (use-local-map ini-mode-map)

  (setq font-lock-maximum-decoration t)

  ;; Set up comments
  (make-local-variable 'comment-start)
  (make-local-variable 'comment-end)
  (make-local-variable 'comment-start-skip)

  ;; Make a menu keymap (with a prompt string)
  ;; and make it the menu bar item's definition.
;   (define-key ini-mode-map [menu-bar] (make-sparse-keymap))
;   (define-key ini-mode-map [menu-bar ini]
;     (cons "ini" (make-sparse-keymap "ini")))

  (setq comment-start "#"
 	comment-end   ""
 	comment-start-skip "#[ \t]*")

  (setq mode-name "INI"
	major-mode 'ini-mode)
  (run-hooks 'ini-mode-hook))

;; Make ini-mode the default mode for ini source code buffers.
;;;###autoload
(setq auto-mode-alist
      (append
       (list
	'("\\.php\\.ini\\'" . ini-mode)
	'("\\.ini\\.php\\'" . ini-mode)
	'("\\.ini\\.append\\'" . ini-mode)
	'("\\.ini\\.append\\.php\\'" . ini-mode)
	'("\\.ini\\'" . ini-mode))
       auto-mode-alist))

;; Set up font locking
(defconst ini-font-lock-keywords-1 nil
  "Subdued level highlighting for ini mode.")

(defconst ini-font-lock-keywords-2 nil
  "Medium level highlighting for ini mode.")

(defconst ini-font-lock-keywords-3 nil
  "Gauchy level highlighting for ini mode.")

(defvar ini-font-lock-keywords ini-font-lock-keywords-3
  "Default expressions to highlight in ini mode.")


;; font-lock-add-keywords is only defined in GNU Emacs
(if (not (string-match "XEmacs" emacs-version))
    (font-lock-add-keywords 'ini-mode ini-font-lock-keywords))

(provide 'ini-mode)
